<?php

namespace App\Http\Controllers;

use PDF;
use App\Cliente;
use App\Producto;
use App\Cotizacion;
use App\SalidaAlmacen;
use App\IngresoAlmacen;
use App\OrdenProduccion;
use App\CotizacionProductos;
use Illuminate\Http\Request;
use App\MateriaPrimaHistorial;
use App\OrdenProduccionParcial;
use App\InventarioMateriaPrimas;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\InventarioMateriaPrimasSalida;

class InventarioController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $opParciales = OrdenProduccionParcial::where('status', 1)->with('op')->get();
        return view('inventario.dashboard', compact('opParciales'));
    }

    public function inventarioHistorial(Request $request)
    {
        // Validar los datos recibidos
        $validatedData = $request->validate([
            '*.codigo_producto' => 'required|string|max:255',
            '*.descripcion_producto' => 'required|string|max:255',
            '*.demanda' => 'required|numeric|min:0',
            '*.inventario' => 'required|numeric|min:0',
            '*.week' => 'required|integer|min:1|max:52',
            '*.year' => 'required|integer|min:2000|max:2100',
            '*.unidad_medida_id' => 'required|integer|exists:productos_medida,id',
        ]);

        DB::beginTransaction();
        try {
            // Guardar los datos en la base de datos
            foreach ($validatedData as $item) {
                MateriaPrimaHistorial::create($item);
            }

            DB::commit();

            // Responder con éxito
            return response()->json([
                'message' => 'Reporte de inventario guardado correctamente.',
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            // Manejar errores y responder con un mensaje de error
            return response()->json([
                'message' => 'Ocurrió un error al guardar el reporte de inventario.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function inventarioDemandaDetalle(Request $request)
    {
        try {
            // Decodificar la cadena JSON a array
            $keywords = json_decode($request->keywords, true);

            // Validar que se obtuvo un array
            if (!is_array($keywords)) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'keywords' => ['The keywords must be an array.']
                ]);
            }

            // Opcional: validar que cada elemento sea string
            foreach ($keywords as $keyword) {
                if (!is_string($keyword)) {
                    throw \Illuminate\Validation\ValidationException::withMessages([
                        'keywords' => ['Cada keyword debe ser un string.']
                    ]);
                }
            }

            $response = $this->getDemandaProductoDetalle($keywords);
            return response()->json($response);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'message' => 'The given data was invalid.',
                'errors' => $e->errors()
            ], 422);
        }
    }

    public function inventarioDetalle(Request $request)
    {
        // Decodificar la cadena JSON a array
        $keywords = json_decode($request->keywords, true);

        // Validar que se obtuvo un array
        if (!is_array($keywords)) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'keywords' => ['The keywords must be an array.']
            ]);
        }

        // Opcional: validar que cada elemento sea string
        foreach ($keywords as $keyword) {
            if (!is_string($keyword)) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'keywords' => ['Cada keyword debe ser un string.']
                ]);
            }
        }

        $response = $this->getInventarioProductoDetalle($keywords);
        return response()->json($response);
    }

    public function getDemandaProductoDetalle($keywords)
    {
        $detalle_demanda = DB::table('orden_produccion as op')
            ->join('productos as p', 'op.id_producto', '=', 'p.id')
            ->leftJoin(DB::raw('(
                SELECT id_op, SUM(cantidad) AS total_parcialidades
                FROM orden_produccion_parcial
                GROUP BY id_op
            ) as opp'), 'op.id', '=', 'opp.id_op')
            ->select(
                'op.id as orden_id',
                'p.nombre as producto',
                'op.cantidad as cantidad_total',
                DB::raw('COALESCE(opp.total_parcialidades, 0) as total_parcialidades'),
                DB::raw('p.factor_conversion_kilos as factor_conversion'),
                DB::raw('
                    CASE
                        WHEN op.STATUS = 3 THEN (op.cantidad - COALESCE(opp.total_parcialidades, 0)) * p.factor_conversion_kilos
                        ELSE op.cantidad * p.factor_conversion_kilos
                    END
                AS Total_KGS_Procesar')
            )
            ->whereNotIn('op.STATUS', [4, 99, 5])
            ->where(function ($query) use ($keywords) {
                foreach ($keywords as $keyword) {
                    $query->orWhere('p.nombre', 'LIKE', "%{$keyword}%");
                }
            })
            ->get();

        return $detalle_demanda;
    }

    // Función para calcular el inventario de un producto usando varias keywords
    public function getInventarioProductoDetalle($keywords)
    {
        // Generar la condición dinámica para las keywords
        $conditions = implode(" OR ", array_map(function ($keyword) {
            return "p.nombre LIKE '%{$keyword}%'";
        }, $keywords));

        // Ejecutar la consulta y obtener el detalle de inventario por producto
        $detalle_inventario = DB::table(DB::raw("(
        SELECT
            p.nombre,
            SUM(inv.total) AS total_por_producto,
            p.factor_conversion_kilos as factor_conversion
        FROM inventario_materias_primas inv
        JOIN productos p ON inv.id_producto = p.id
        WHERE inv.total > 0
          AND ($conditions)
        GROUP BY inv.id_producto
    ) AS subquery"))
            ->get();

        return $detalle_inventario;
    }

    public function ordenProduccionRecibido(OrdenProduccionParcial $ordenProduccionP)
    {
        $opParciales = OrdenProduccionParcial::where('id', $ordenProduccionP->id)->first();
        $op = OrdenProduccion::where('id', $ordenProduccionP->id_op)->with('cliente')->with('producto')->first();
        return view('inventario.recibir_orden_produccion', compact('op', 'opParciales'));
    }
    public function salidaAlamacen()
    {
        $salidasAlmacen = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', '!=', 999999)
            ->where('id_op', '!=', 9999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $salidasAlmacenInventario = IngresoAlmacen::where('status', 1)
            ->where('id_cotizacion_producto', '>', 0)
            ->with('cliente')
            ->with('producto')
            ->get();


        //dd($salidasAlmacenInventario);
        return view('inventario.dashboard_salida', compact('salidasAlmacen', 'salidasAlmacenInventario'));
    }


    public function HistorialDespacho()
    {
        $salidasAlmacen = IngresoAlmacen::where('id_cliente', '!=', 999999)
            ->where('id_op', '!=', 9999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $salidasAlmacenInventario = IngresoAlmacen::where('id_cotizacion_producto', '>', 0)
            ->with('cliente')
            ->with('producto')
            ->get();


        //dd($salidasAlmacenInventario);
        //dd($salidasAlmacen);
        return view('inventario.historial_despacho', compact('salidasAlmacen', 'salidasAlmacenInventario'));
    }

    public function archivarMuestra($idEnvio)
    {
        try {
            $envioMuestra = SalidaAlmacen::where('id', $idEnvio)->first();
            $envioMuestra->estatus =  99;
            $envioMuestra->update();

            $arrResponse = [
                'message' => 'Se archivo correctamente el envio de muestra'
            ];

            return redirect()->route('inventario.dashboard_envios')->with($arrResponse);
        } catch (\Throwable $th) {
            $arrResponse = [
                'message' => 'Error al archivar el envio de muestra: ' . $th->getMessage()
            ];
            var_dump($th->getMessage());
        }
    }

    public function archivarSalida(Request $request)
    {

        $salida = IngresoAlmacen::where('id', $request->id)->first();
        $salida->observaciones =  $request->mensaje;
        $salida->status =  77;
        $salida->update();

        $ordenes = OrdenProduccionParcial::where('id_op', $salida->id_op)->get();
        if ($ordenes) {
            foreach ($ordenes as $orden) {
                $orden->estatus =  77;
                $orden->obseervacion_almacen = $request->mensaje;
                $orden->update();
            }
            $cotizacion_producto = CotizacionProductos::where('id', $salida->id_cotizacion_producto)->first();
            $cotizacion_producto->estatus = 77;
            $cotizacion_producto->update();
        }


        $arrResponse = [
            'message' => 'Se archivo correctamente la cotización y salida de almacen'
        ];
        return redirect()->route('inventario.dashboard_salida')->with($arrResponse);
    }

    public function reverseEnvio(Request $request)
    {
        try {
            DB::beginTransaction();
            $salida = SalidaAlmacen::find($request->envio_id);
            $op = OrdenProduccion::where('id', $salida->id_op)->first();
            if ($op) {
                $op->cantidad_entregada = $op->cantidad_entregada - $salida->cantidad;
                $op->status =  3;
                $op->update();
            }

            $salida->delete();

            $arrResponse = [
                'message' => 'Se revirtio correctamente la salida de almacen'
            ];

            DB::commit();

            return response($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            $arrResponse = [
                'message' => 'Error al revertir la salida de almacen: ' . $th->getMessage() . ' ' . $th->getLine() . ' ' . $th->getFile()
            ];
            return response($arrResponse, 500);
        }
    }

    public function dashboardInventario()
    {
        $inventario = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', 999999)
            ->with('producto')
            ->groupBy('id_producto')
            ->selectRaw('*,sum(cantidad) as total')
            ->get();
        $clientes = Cliente::orderBy('nombre', 'asc')->get();
        return view('inventario.dashboard_inventario', compact('inventario', 'clientes'));
    }
    public function dashboardInventarioMateriaPrima()
    {
        $inventario = InventarioMateriaPrimas::select('id_producto', DB::raw('SUM(total) as total'))
            ->groupBy('id_producto')
            ->havingRaw('SUM(total) > 0')
            ->get();
        return view('inventario.dashboard_inventario_prima', compact('inventario'));
    }
    public function dashboardInventarioMateriaPrimaSalidas()
    {
        $inventario = InventarioMateriaPrimasSalida::all();
        return view('inventario.dashboard_inventario_prima_salida', compact('inventario'));
    }
    public function salidaAlmacenFactura(Request $request)
    {
        $totalProduccion = $request->cantidad - $request->produccion;
        $x = new SalidaAlmacen();
        $x->id_usuario =  Auth::id();
        $x->id_op = $request->id_op;
        $x->id_cliente = $request->id_cliente;
        $x->id_producto = $request->id_producto;
        $x->cantidad = $request->cantidad;
        $x->estatus = 1;
        $x->observaciones = $request->observaciones ?? '';
        $x->save();

        $op = OrdenProduccion::where('id', $request->id_op)->first();
        $cantidadTotalEntregada = $op->cantidad_entregada + $request->cantidad;
        $op->cantidad_entregada = $cantidadTotalEntregada;
        $op->update();
        //Quitar de inventario

        while ($totalProduccion > 0) {
            $inventario = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->first();

            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();

                $op = OrdenProduccion::where('id', $request->id_op)->first();
                $op->cantidad_inventario = $op->cantidad_inventario + $totalProduccion;
                $op->update();
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();

                $op = OrdenProduccion::where('id', $request->id_op)->first();
                $op->cantidad_inventario = $op->cantidad_inventario + $inventario->cantidad;
                $op->update();
            }
        }
        $arrResponse = [
            'message' => 'Salida de almacen correctamente'
        ];
        return response($x);
    }
    public function salidaAlmacenFacturacion(Request $request)
    {
        $op = SalidaAlmacen::where('id', $request->id_salida)->with('cliente')->first();
        $cantidad_cotizacion = $op->cotizacion->cantidad ?? 0;
        $cantidad_op = $op->op->cotizaciones_p->cantidad ?? 0;
        $cantidad = $op->cantidad ?? 0;
        if ($cantidad_cotizacion <= $cantidad && $cantidad_cotizacion != 0) {
            $op->estatus = 6;
        } elseif ($cantidad_op <= $cantidad && $cantidad_op != 0) {
            $op->estatus = 6;
        } else {
            $op->estatus = 2;
        }
        $op->update();
        $id_salida = $request->id_salida;
        $no_cotizacion = $op->id_cotizacion > 0 ? $op->id_cotizacion : $op->op->id_cotizacion_producto;
        $info = [
            'idSalida' => $id_salida,
            'idCotizacion' => $no_cotizacion,
            'nombreCliente' => $op->cliente->referencia
        ];

        Mail::send('email.envio_pendiente_factura', compact('info'), function ($message) use ($id_salida) {
            $message->from('info@laravelrp.com', 'Concisa - Despacho realizado listo para facturar ' . $id_salida);
            //$message->to('kicas48@gmail.com');
            $message->to('correo@gmail.com');
            //$message->to('esmeralda@concisa.com.gt');
            $message->subject('Despacho realizado listo para facturar - ' . $id_salida);
        });
        return response($op);
    }
    public function enviosAlamacen()
    {
        $salidas = SalidaAlmacen::where('estatus', 1)
            ->where('id_cliente', '>', 0)
            ->where('id_producto', '>', 0)
            ->with(['cliente', 'producto', 'producto.medida', 'cliente.info_ejecutivo'])
            ->with(['op.cotizaciones_p.cotizacion' => function ($query) {
                $query->select('id', 'muestra'); // Selecciona solo el campo necesario
            }])
            ->get();

        $salidas->each(function ($salida) {
            $salida->mostrarMuestra = $salida->op?->cotizaciones_p?->cotizacion?->muestra ?? false;
        });
        //dd($salidas);
        return view('inventario.dashboard_envios', compact('salidas'));
    }
    public function generarEnvio($idEnvio)
    {

        $salidas = SalidaAlmacen::where('id', $idEnvio)
            ->with('cliente')
            ->with('producto')
            ->first();
        //dd($salidas);
        $info_envio = [
            'id' => $salidas->id,
            'referencia' => $salidas->cliente->referencia,
            'dia' => date("d"),
            'mes' => date("m"),
            'ano' => date("Y"),
            'cliente' => $salidas->cliente->nombre,
            'cantidad' => $salidas->cantidad,
            'medida' => $salidas->producto->medida->nombre,
            'producto' => $salidas->producto->nombre,
            'descripcion' => $salidas?->op?->cotizaciones_p?->descripcion ?? '',
        ];
        $pdf = PDF::loadView('inventario.generar_envio', compact('info_envio'));
        return $pdf->download('envio - ' . $salidas->id . '.pdf');
        dd();
    }

    public function salidaAlmacenFacturaInventario(Request $request)
    {
        $totalProduccion = $request->cantidad;
        $noActualizar = 1;
        while ($totalProduccion > 0) {
            $inventario = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->first();
            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->id_cliente =  $request->id_cliente;
                    $x->id_producto =  $request->id_producto;
                    $x->cantidad = $totalProduccion;
                    $x->estatus = 1;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->estatus = 1;
                    $x->id_producto =  $request->id_producto;
                    $x->id_cliente =  $request->id_cliente;
                    $x->cantidad = $request->cantidad;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }


                $noActualizar = 2;
            }
        }


        return response($x);
    }



    public function salidaAlmacenFacturaInventarioEliminar(Request $request)
    {
        $totalProduccion = $request->cantidad;
        $inventario = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', 999999)
            ->where('cantidad', '>', 0)
            ->where('id_producto', $request->id_producto)
            ->first();
        $noActualizar = 1;

        while ($totalProduccion > 0) {
            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->id_cliente =  9999999;
                    $x->cantidad = $totalProduccion;
                    $x->estatus = 3; //Desperdicio
                    $x->id_producto = $request->id_producto;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->estatus = 3; //Desperdicio
                    $x->id_cliente =  9999999;
                    $x->cantidad = $request->cantidad;
                    $x->id_producto = $request->id_producto;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $noActualizar = 2;
            }
        }
        $producto = Producto::where('id', $request->id_producto)->with('medida')->first();
        $info = [
            'producto' => $producto->nombre,
            'cantidad' => $request->cantidad,
            'observacion' => $request->observaciones ?? '',
            'medida' => $producto->medida->nombre
        ];
        $idOP = $inventario->id;
        Mail::send('email.inventario_eliminar', compact('info'), function ($message) use ($idOP) {
            $message->from('info@laravelrp.com', 'Concisa - Desechar Producto ' . $idOP);
            $message->to('kicas48@gmail.com');
            $message->to('anylopeza3@gmail.com');
            $message->to('erickfernando85@gmail.com');
            $message->subject('Desechar producto - ' . $idOP);
        });
        return response($x);
    }
    public function salidaAlmacenInventario(Request $request)
    {
        //dd('entro');
        $salidaAlmacen = IngresoAlmacen::where('id', $request->id_inventario)->first();
        $cantidad_solicitud = floatval($request->cantidad);
        $cantidad_orden = floatval($salidaAlmacen->cantidad);
        if ($cantidad_solicitud == $cantidad_orden) {
            $salidaAlmacen->id_cliente = 999999;
            $salidaAlmacen->update();
        } else {
            $op = new IngresoAlmacen();
            $op->id_usuario =  Auth::id();
            $op->id_cliente = 999999;
            $op->id_producto = $request->id_producto;
            $op->cantidad = $cantidad_solicitud;
            $op->id_op = $request->id_op;
            $op->id_op_parcial = 0;
            $op->observaciones = $request->observaciones ?? '';
            $op->status = 1;
            $op->save();

            $salidaAlmacen->cantidad = $cantidad_orden - $cantidad_solicitud;
            $salidaAlmacen->update();
        }
        return response(200);
    }
    public function salidaAlmacenTerminar(Request $request)
    {
        if ($request->cantidad > 0) {

            if ($request->cantidad <= $request->produccion) {
                echo 'Entro1';
                $x = new SalidaAlmacen();
                $x->id_usuario =  Auth::id();
                $x->id_op = $request->id_op;
                $x->id_cliente = $request->id_cliente;
                $x->cantidad = $request->cantidad;
                $x->id_producto = $request->id_producto;
                $x->estatus = 1;
                $x->observaciones = $request->observaciones ?? '';
                $x->save();
            }
            $op = OrdenProduccion::where('id', $request->id_op)->first();
            $op->cantidad_entregada = $op->cantidad_entregada + $request->cantidad;
            $op->status = 5;
            $op->update();
            echo 'Entro2';
        } else {
            echo 'Entro3';
            $op = OrdenProduccion::where('id', $request->id_op)->first();
            $op->status = 5;
            $op->update();
        }
        $arrResponse = [
            'message' => 'Salida de almacen correctamente'
        ];
        return response(200);
    }
    public function salidaAlmacenInventarioTerminar(Request $request)
    {
        $cantidad = floatval($request->cantidad);
        $inventario = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', 999999)
            ->where('cantidad', '>', 0)
            ->where('id_producto', $request->id_producto)
            ->get();
        foreach ($inventario as $in) {
            if ($cantidad > 0) {
                $cantidad_inventario = floatval($in->cantidad);
                if ($cantidad <= $cantidad_inventario) {
                    $resultado = $cantidad_inventario - $cantidad;
                    $cantidad = 0;
                    $inventarioActualizar = IngresoAlmacen::where('id', $in->id)->first();
                    $inventarioActualizar->cantidad = $resultado;
                    $inventarioActualizar->update();
                } else {
                    $cantidad = $cantidad - $cantidad_inventario;
                    $inventarioActualizar = IngresoAlmacen::where('id', $in->id)->first();
                    $inventarioActualizar->cantidad = 0;
                    $inventarioActualizar->update();
                }
            }
        }
        if ($cantidad == 0) {
            $ingresoActualizar = IngresoAlmacen::where('id', $request->id_salida)->first();
            $ingresoActualizar->status = 2;
            $ingresoActualizar->update();

            $x = new SalidaAlmacen();
            $x->id_usuario =  Auth::id();
            $x->id_cotizacion = $request->id_cotizacion;
            $x->cantidad = $request->cantidad;
            $x->id_producto = $request->id_producto;
            $x->id_cliente = $ingresoActualizar->id_cliente;
            $x->estatus = 1;
            $x->observaciones = $request->observaciones ?? '';
            $x->save();
            $arrResponse = [
                'message' => 'Salida de almacen correctamente'
            ];
        } else {
            $arrResponse = [
                'message' => 'No se ingreso la cantidad'
            ];
        }

        return response(200);
    }
    public function historialProducto($idProducto)
    {
        $productos = IngresoAlmacen::where('id_producto', $idProducto)->where('status', 1)->with('producto')->get();
        return view('inventario.dashboard_producto', compact('productos'));
    }
    public function ingresoAlmacen(Request $request)
    {
        $op = new IngresoAlmacen();
        $op->id_usuario =  Auth::id();
        $op->id_cliente = $request->id_cliente;
        $op->id_producto = $request->id_producto;
        $op->cantidad = $request->cantidad;
        $op->id_op = $request->id_op;
        $op->id_op_parcial = $request->id_op_parcial;
        $op->fecha_recibida = date("Y-m-d", strtotime($request->fecha_recibida));
        $op->id_cotizacion = $request->id_cotizacion;
        $op->observaciones = $request->observaciones ?? '';
        $op->status = 1;
        $op->save();

        $op = OrdenProduccionParcial::where('id', $request->id_op_parcial)->first();
        $op->status = 99;
        $op->update();

        $arrResponse = [
            'message' => 'Ingreso almacen correctamente'
        ];
        return redirect()->route('inventario.dashboard')->with($arrResponse);
    }
    public function ingresoAlmacenAjax(Request $request)
    {
        $opParciales = OrdenProduccionParcial::where('id', $request->idOrden)->with('op')->first();
        $ingreso_almacen = new IngresoAlmacen();
        $ingreso_almacen->id_usuario =  Auth::id();
        $ingreso_almacen->id_cliente = $opParciales->op->id_cliente;
        $ingreso_almacen->id_producto = $opParciales->id_producto;
        $ingreso_almacen->cantidad = $opParciales->cantidad;
        $ingreso_almacen->id_op = $opParciales->op->id;
        $ingreso_almacen->id_op_parcial = $opParciales->id;
        $ingreso_almacen->observaciones = $opParciales->op->observaciones ?? '';
        $ingreso_almacen->status = 1;
        $ingreso_almacen->save();

        $opParciales->status = 99;
        $opParciales->update();

        $arrResponse = [
            'message' => 'Ingreso almacen correctamente'
        ];
        return response($ingreso_almacen);
    }
    public function dashboardArchivo()
    {
        //
        $salidasAlmacen = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', '!=', 999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $anuladasAlmacen = IngresoAlmacen::where('status', 77)
            ->where('id_cliente', '!=', 999999)
            ->with('cliente')
            ->with('producto')
            ->get();

        return view('inventario.dashboard_archivo', compact('salidasAlmacen', 'anuladasAlmacen'));
    }
    public function ajusteInventario()
    {

        return view('inventario.ajuste_inventario');
    }
    public function ajusteInventarioPrima()
    {
        return view('inventario.ajuste_inventario_prima');
    }
    public function ajusteInventarioGuardar(Request $request)
    {
        if ($request->tipo_solicitud == 1) {
            $ingreso_almacen = new IngresoAlmacen();
            $ingreso_almacen->id_usuario =  Auth::id();
            $ingreso_almacen->id_cliente = $request->id_cliente;
            $ingreso_almacen->id_producto = $request->id_producto;
            $ingreso_almacen->cantidad = $request->cantidad;
            $ingreso_almacen->id_op = 999999;
            $ingreso_almacen->id_op_parcial = 999999;
            $ingreso_almacen->observaciones = $request->observaciones ?? '';
            $ingreso_almacen->status = 1;
            $ingreso_almacen->save();
        } else {
            $totalProduccion = $request->cantidad;
            $inventarios = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->get();
            foreach ($inventarios as $inv) {
                if ($inv->cantidad >= $totalProduccion) {
                    if ($totalProduccion > 0) {
                        $rebajar = IngresoAlmacen::where('id', $inv->id)->first();
                        $rebajar->cantidad = floatval($rebajar->cantidad) - $totalProduccion;
                        $rebajar->observacion_rebaja = 'Cantidad=' . $totalProduccion . ' ' . $request->observaciones;
                        $rebajar->update();
                        $totalProduccion = 0;
                    }
                } else {
                    $rebajar = IngresoAlmacen::where('id', $inv->id)
                        ->first();
                    $totalProduccion = $totalProduccion - floatval($rebajar->cantidad);
                    $rebajar->cantidad = 0;
                    $rebajar->observacion_rebaja = 'Cantidad=' . $totalProduccion . ' ' . $request->observaciones;
                    $rebajar->update();
                }
            }
        }
        $arrResponse = [
            'message' => 'Inventario actualizado'
        ];
        return redirect()->route('inventario.dashboard_inventario')->with($arrResponse);
    }
    public function ajusteInventarioPrimaGuardar(Request $request)
    {
        if ($request->tipo_solicitud == 1) {
            $ingreso_almacen = new InventarioMateriaPrimas();
            $ingreso_almacen->id_usuario =  Auth::id();
            $ingreso_almacen->id_producto = $request->id_producto;
            $ingreso_almacen->cantidad = $request->cantidad;
            $ingreso_almacen->total = $request->total;
            $ingreso_almacen->observaciones = $request->observaciones ?? '';
            $ingreso_almacen->save();
            $arrResponse = [
                'message' => 'Inventario actualizado'
            ];
            return redirect()->route('inventario.dashboard_inventario_materia_prima')->with($arrResponse);
        } else {
            $total_quitar = floatval($request->total);
            $inventarios = InventarioMateriaPrimas::where('id_producto', $request->id_producto)
                ->where('total', '>', 0)
                ->get();

            foreach ($inventarios as $inv) {
                if ($inv->total >= $total_quitar) {
                    $rebajar = InventarioMateriaPrimas::where('id', $inv->id)
                        ->first();
                    $rebajar->total = floatval($rebajar->total) - $total_quitar;
                    $rebajar->update();
                    $total_quitar = 0;
                } else {
                    $rebajar = InventarioMateriaPrimas::where('id', $inv->id)
                        ->first();
                    $total_quitar = $total_quitar - floatval($rebajar->total);
                    $rebajar->total = 0;
                    $rebajar->update();
                }
            }
            if ($total_quitar == 0) {
                $ingreso_almacen = new InventarioMateriaPrimasSalida();
                $ingreso_almacen->id_usuario =  Auth::id();
                $ingreso_almacen->id_producto = $request->id_producto;
                $ingreso_almacen->cantidad = $request->cantidad;
                $ingreso_almacen->total = $request->total;
                $ingreso_almacen->observaciones = $request->observaciones ?? '';
                $ingreso_almacen->save();
            } else {
                $ingreso_almacen = new InventarioMateriaPrimasSalida();
                $ingreso_almacen->id_usuario =  Auth::id();
                $ingreso_almacen->id_producto = $request->id_producto;
                $ingreso_almacen->cantidad = $request->cantidad;
                $ingreso_almacen->total = floatval($request->total) - $total_quitar;
                $ingreso_almacen->observaciones = $request->observaciones ?? '';
                $ingreso_almacen->save();
            }

            $arrResponse = [
                'message' => 'Inventario actualizado'
            ];
            return redirect()->route('inventario.dashboard_inventario_materia_prima')->with($arrResponse);
        }
    }
}
