<?php

namespace App\Http\Controllers;

use App\Compra;
use Illuminate\Http\Request;
use App\MateriaPrimaHistorial;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ArchivoReporteController extends Controller
{
    public function index()
    {
        // Definir los productos con su descripción y palabras clave (keywords)
        $productos = [
            [
                'descripcion' => 'BOPP TSE 30 (50 Y 100 CMS)',
                'keywords'    => ['TSE30']
            ],
            // [
            //     'descripcion' => 'BOPP TSE 35',
            //     'keywords'    => ['TSE35']
            // ],
            // [
            //     'descripcion' => 'BOPP TSE 37',
            //     'keywords'    => ['TSE37']
            // ],
            // [
            //     'descripcion' => 'BOPP TSE 40',
            //     'keywords'    => ['TSE40']
            // ],
            [
                'descripcion' => 'BOPP TSE 35, 37 Y 40',
                'keywords'    => ['TSE35', 'TSE37', 'TSE40']
            ],
            [
                'descripcion' => 'CARTON SBS',
                'keywords'    => ['LIENZO SBS', 'BOBINA CARTON']
            ],
            [
                'descripcion' => 'PLASTIPEL (25 CMS)',
                'keywords'    => ['PLASTIPEL', 'PLASTIPET']
            ],
            [
                'descripcion' => 'BOPP TSE 30 (BLANCO PERLADO)',
                'keywords'    => ['TSE30 BLANCO PERLADO', 'PERLADO']
            ],
            [
                'descripcion' => 'PAPEL ANTIGRASA BLANCO',
                'keywords'    => ['ANTIGRASA BLANCO']
            ],
            [
                'descripcion' => 'PAPEL ANTIGRASA KRAFT',
                'keywords'    => ['ANTIGRASA KRAFT']
            ],
            [
                'descripcion' => 'BAKING PAPER',
                'keywords'    => ['COCINA BLANCO']
            ]
        ];

        // Mapear cada producto a una estructura con demanda e inventario usando las keywords
        $materias = collect($productos)->map(function ($producto) {
            return (object)[
                'descripcion' => $producto['descripcion'],
                'demanda'     => $this->getDemandaProducto($producto['keywords']),
                'inventario'  => $this->getInventarioProducto($producto['keywords']),
                'keywords'    => $producto['keywords']
            ];
        });

        // Retornar vista con los datos
        return view('inventario.archivo_reporte', compact('materias'));
    }


    // Función para calcular la demanda de un producto usando varias keywords
    public function getDemandaProducto($keywords)
    {
        $demanda_producto = DB::table('orden_produccion as op')
            ->join('productos as p', 'op.id_producto', '=', 'p.id')
            ->leftJoin(DB::raw('(
            SELECT id_op, SUM(cantidad) AS total_parcialidades
            FROM orden_produccion_parcial
            GROUP BY id_op
        ) as opp'), 'op.id', '=', 'opp.id_op')
            ->selectRaw('SUM(
            CASE
                WHEN op.STATUS = 3 THEN (op.cantidad - COALESCE(opp.total_parcialidades, 0)) * p.factor_conversion_kilos
                ELSE op.cantidad * p.factor_conversion_kilos
            END
        ) AS Total_KGS_Procesar')
            ->whereNotIn('op.STATUS', [4, 99, 5])
            ->where(function ($query) use ($keywords) {
                foreach ($keywords as $keyword) {
                    $query->orWhere('p.nombre', 'LIKE', "%{$keyword}%");
                }
            })
            ->value('Total_KGS_Procesar') ?? 0;

        return $demanda_producto;
    }

    // Función para calcular el inventario de un producto usando varias keywords
    public function getInventarioProducto($keywords)
    {
        // Generar la condición dinámica para las keywords
        $conditions = implode(" OR ", array_map(function ($keyword) {
            return "p.nombre LIKE '%{$keyword}%'";
        }, $keywords));

        $inventario_producto = DB::table(DB::raw("(
        SELECT
            p.nombre,
            SUM(inv.total) AS TOTAL
        FROM inventario_materias_primas inv
        JOIN productos p ON inv.id_producto = p.id
        WHERE inv.total > 0
          AND ($conditions)
        GROUP BY inv.id_producto
    ) AS subquery"))
            ->selectRaw('SUM(TOTAL) AS Total_Global')
            ->value('Total_Global') ?? 0;

        return $inventario_producto;
    }

    public function importaciones()
    {
        $compras = Compra::where('tipo_compra', 1)->where('estado', 3)->orderBy('id', 'desc')
            ->with('proveedor')
            ->with('detalle')
            ->with('pagos')
            ->with('envio')
            ->get();
        return view('reportes.gerencia.compras.importaciones', compact('compras'));
    }

    public function detalleImportaciones($compra_id)
    {
        $compra = Compra::where('id', $compra_id)
            ->with('proveedor')
            ->with('detalle')
            ->with('pagos')
            ->with('envio')
            ->first();

        $compra_eta = $compra->envio->first() ?? new CompraEnvio();
        if ($compra->tipo_cambio === null || $compra->tipo_cambio < 1) {
            $compra->tipo_cambio = 1;
        }
        $compra->incoterm = match ($compra->incoterm) {
            1 => 'EXW',
            2 => 'FOB',
            3 => 'CIF',
            default => 'DESCONOCIDO'
        };
        $tipo_cambio = $compra->tipo_cambio;
        $total_factura = $compra->total_factura; //Dolares
        $total_dai = $compra->total_dai ?? 0; //Dolares
        $total_flete_seguro = $compra->total_flete_seguro; //Dolares

        $prorrateo_datos = null;
        if (count($compra->pagos) > 0) {
            $prorrateo_datos = new \stdClass();
            // Paso 1
            //Datos a obtener: pago en dolares a naviera, total de gastos locales sin iva y con iva, total de importacion (valor factura+valor flete+DAI+Pago en dolares naviera + total gastos locales, todo en quetzales).

            //Total de factura + flete y seguro
            $prorrateo_datos->valor_cif_factura = ($total_factura + $total_flete_seguro);
            $prorrateo_datos->valor_cif_factura_quetzales = ($prorrateo_datos->valor_cif_factura * $tipo_cambio) == 0 ? 1 : ($prorrateo_datos->valor_cif_factura * $tipo_cambio);

            //Total de dai en dolares y quetzales (el valor se guarda en dolares)
            $prorrateo_datos->total_dai = $total_dai ?? 0;
            $prorrateo_datos->total_dai_quetzales = $total_dai;
            //Total de pago naviera (tipo de pago entre 2 y 6)
            $prorrateo_datos->pago_naviera_dolares = $compra->pagos->where('tipo_pago', 2)->sum('monto') ?? 0;
            $prorrateo_datos->pago_naviera_dolares += $compra->pagos->where('tipo_pago', 3)->sum('monto') ?? 0;
            $prorrateo_datos->pago_naviera_quetzales = $prorrateo_datos->pago_naviera_dolares * $tipo_cambio;

            //Total de gastos locales
            $gastosLocales = $compra->pagos->where('tipo_pago', '>', 3);
            $gastosLocalesSinIvaQuetzales = $gastosLocales->where('iva', 0)->where('moneda_id', 1)->sum('monto');
            $gastosLocalesSinIvaDolaresAQuetzales = $gastosLocales->where('iva', 0)->where('moneda_id', 2)->map(function ($pago) use ($tipo_cambio) {
                return $pago->monto * $tipo_cambio;
            })->sum();

            // Gastos Locales con IVA
            $gastosLocalesConIvaQuetzales = $gastosLocales->where('iva', 1)->where('moneda_id', 1)->sum('monto') / 1.12 ?? 0;
            $gastosLocalesConIvaDolaresAQuetzales = $gastosLocales->where('iva', 1)->where('moneda_id', 2)->map(function ($pago) use ($tipo_cambio) {
                return ($pago->monto * $tipo_cambio) / 1.12;
            })->sum() ?? 0;

            $prorrateo_datos->gastos_locales_sin_iva = $gastosLocalesSinIvaQuetzales + $gastosLocalesSinIvaDolaresAQuetzales + $gastosLocalesConIvaQuetzales + $gastosLocalesConIvaDolaresAQuetzales;
            $prorrateo_datos->gastos_locales_sin_iva_dolares = $prorrateo_datos->gastos_locales_sin_iva / $tipo_cambio;

            // var_dump($prorrateo_datos);
            // Paso 2
            $prorrateo_datos->total_importacion_quetzales = $prorrateo_datos->valor_cif_factura_quetzales + $prorrateo_datos->total_dai_quetzales + $prorrateo_datos->pago_naviera_quetzales + $prorrateo_datos->gastos_locales_sin_iva;
            // $prorrateo_datos->total_importacion_dolares = $prorrateo_datos->valor_cif_factura + $prorrateo_datos->total_dai + $prorrateo_datos->pago_naviera_dolares + $prorrateo_datos->gastos_locales_sin_iva_dolares;

            //Paso 3 coeficientes
            $prorrateo_datos->coeficiente1 = number_format($prorrateo_datos->pago_naviera_quetzales / $prorrateo_datos->valor_cif_factura_quetzales, 6);
            $prorrateo_datos->coeficiente2 = number_format($prorrateo_datos->gastos_locales_sin_iva / $prorrateo_datos->valor_cif_factura_quetzales, 6);
        }

        return view('reportes.gerencia.compras.importaciones-detalle', compact('compra', 'prorrateo_datos', 'compra_eta'));
    }

    public function upload(Request $request)
    {
        try {
            $request->validate([
                'files.*' => 'required|file|mimes:xlsx,xls',
                'year' => 'required|integer',
                'month' => 'required|integer|between:1,12',
            ]);

            // Crear la carpeta si no existe
            $folderPath = "public/{$request->year}/{$request->month}";
            if (!Storage::exists($folderPath)) {
                Storage::makeDirectory($folderPath);
            }

            // Procesar cada archivo
            foreach ($request->file('files') as $file) {
                $filePath = $file->storeAs($folderPath, $file->getClientOriginalName());
                $filePathFromStorage = str_replace('public/', 'storage/', $filePath);

                // Insertar en la base de datos
                DB::table('archivos_reportes')->insert([
                    'file_name' => $file->getClientOriginalName(),
                    'file_path' => $filePathFromStorage,
                    'year' => $request->year,
                    'month' => $request->month,
                ]);
            }

            return redirect()->route('reportes.producto_terminado')->with('success', 'Archivos cargados exitosamente.');
        } catch (\Throwable $th) {
            dd($th->getMessage() . ' ' . $th->getLine());
        }
    }


    public function delete($id)
    {
        $file = DB::table('archivos_reportes')->find($id);
        if ($file) {
            Storage::delete(str_replace('storage/', 'public/', $file->file_path));
            DB::table('archivos_reportes')->where('id', $id)->delete();
        }
        return redirect()->route('reportes.producto_terminado')->with('success', 'Archivo eliminado.');
    }

    public function replace(Request $request, $id)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx',
        ]);

        $file = DB::table('archivos_reportes')->find($id);
        if ($file) {
            Storage::delete(str_replace('storage/', 'public/', $file->file_path));
            $filePath = $request->file('file')->storeAs("public/{$file->year}/{$file->month}", $request->file('file')->getClientOriginalName());
            // Cambiar public/ por storage/ en la ruta de almacenamiento
            $filePathFromStorage = str_replace('public/', 'storage/', $filePath);
            DB::table('archivos_reportes')->where('id', $id)->update([
                'file_name' => $request->file('file')->getClientOriginalName(),
                'file_path' => $filePathFromStorage,
            ]);
        }

        return redirect()->route('reportes.producto_terminado')->with('success', 'Archivo reemplazado.');
    }
}
